import json
from zipfile import ZipFile
from xml.etree import ElementTree
from pathlib import Path

# ----------------------------------------------------------------
# PATH SETUP (script is in card/loot_table)
# ----------------------------------------------------------------
SCRIPT_DIR = Path(__file__).parent

CARD_FOLDER = SCRIPT_DIR / "card"   # card/loot_table/card
RAND_FOLDER = SCRIPT_DIR / "rand"   # card/loot_table/rand
FUNCTION_FOLDER = SCRIPT_DIR.parent / "function" / "admin"  # card/function/admin

CARD_FOLDER.mkdir(exist_ok=True)
RAND_FOLDER.mkdir(exist_ok=True)
FUNCTION_FOLDER.mkdir(parents=True, exist_ok=True)

# Excel file in same folder as script
EXCEL_FILE = SCRIPT_DIR / "cards.xlsx"

# ----------------------------------------------------------------
# DELETE OLD FILES
# ----------------------------------------------------------------
for old in CARD_FOLDER.glob("*.json"):
    old.unlink()
print("Cleared old card JSONs.")

for old in RAND_FOLDER.glob("*.json"):
    old.unlink()
print("Cleared old random table JSONs.")

# ----------------------------------------------------------------
# XLSX PARSING (no external libraries)
# ----------------------------------------------------------------
def load_shared_strings(zipf):
    if "xl/sharedStrings.xml" not in zipf.namelist():
        return []
    with zipf.open("xl/sharedStrings.xml") as f:
        root = ElementTree.parse(f).getroot()
    out = []
    for si in root.findall("{http://schemas.openxmlformats.org/spreadsheetml/2006/main}si"):
        parts = []
        for t in si.iter():
            if t.tag.endswith("t"):
                parts.append(t.text or "")
        out.append("".join(parts))
    return out

def get_cell_value(cell, shared):
    t = cell.get("t")
    v = cell.find("{http://schemas.openxmlformats.org/spreadsheetml/2006/main}v")
    if v is None:
        return ""
    value = v.text
    if t == "s":
        return shared[int(value)]
    return value

def col_to_index(col):
    num = 0
    for c in col:
        num = num * 26 + (ord(c) - 65 + 1)
    return num - 1

def read_excel_simple(path):
    with ZipFile(path) as zipf:
        shared = load_shared_strings(zipf)
        with zipf.open("xl/worksheets/sheet1.xml") as f:
            root = ElementTree.parse(f).getroot()

        rows = []
        for row in root.iter("{http://schemas.openxmlformats.org/spreadsheetml/2006/main}row"):
            vals = {}
            for c in row:
                ref = c.get("r")
                letters = "".join(ch for ch in ref if ch.isalpha())
                idx = col_to_index(letters)
                vals[idx] = get_cell_value(c, shared)
            if vals:
                full = [vals.get(i, "") for i in range(max(vals.keys())+1)]
                rows.append(full)
        return rows

# ----------------------------------------------------------------
# PROCESS EXCEL
# ----------------------------------------------------------------
RARITY_MAP = {
    "Common": ("common", 100),
    "Rare": ("rare", 200),
    "Legendary": ("legendary", 800)
}

NAME_COLOR_MAP = {
    "Common": "white",
    "Rare": "aqua",
    "Legendary": "gold"
}

print(f"Reading: {EXCEL_FILE}")

rows = read_excel_simple(EXCEL_FILE)

all_cards = []
common_cards = []
rare_cards = []
legendary_cards = []

# Skip header row
for i, row in enumerate(rows[1:], start=2):
    if len(row) < 4:
        continue

    display = row[0].strip()
    rarity_raw = row[1].strip()
    ident = row[2].strip()
    desc = row[3].strip()

    if rarity_raw not in RARITY_MAP:
        continue

    rarity_key, price = RARITY_MAP[rarity_raw]
    name_color = NAME_COLOR_MAP[rarity_raw]

    lore = desc.split("|")
    while len(lore) < 4:
        lore.append("")
    lore = lore[:4]

    namespaced = f"card:card/{ident}"
    all_cards.append(namespaced)

    if rarity_raw == "Common":
        common_cards.append(namespaced)
    elif rarity_raw == "Rare":
        rare_cards.append(namespaced)
    elif rarity_raw == "Legendary":
        legendary_cards.append(namespaced)

    # ----------------------------------------------------------------
    # WRITE INDIVIDUAL CARD JSON
    # ----------------------------------------------------------------
    card_json = {
        "pools": [
            {
                "rolls": 1,
                "entries": [
                    {
                        "type": "item",
                        "weight": 1,
                        "name": "minecraft:flow_banner_pattern",
                        "functions": [
                            {"function": "set_count", "count": 1},
                            {
                                "function": "set_components",
                                "components": {
                                    "minecraft:max_stack_size": 1,
                                    "minecraft:custom_data": {
                                        "card_flag": 1,
                                        "card": ident,
                                        "rarity": rarity_key,
                                        "true_price": price
                                    },
                                    "minecraft:item_model": f"cards:{ident}"
                                }
                            },
                            {
                                "function": "set_name",
                                "name": [{
                                    "color": name_color,
                                    "italic": False,
                                    "underlined": True,
                                    "text": display
                                }]
                            },
                            {
                                "function": "set_lore",
                                "lore": [
                                    {"color": "white", "italic": True, "text": f"{rarity_raw} Card"},
                                    {"color": "gray", "italic": False, "text": " "},
                                    {"color": "gray", "italic": False, "text": lore[0]},
                                    {"color": "gray", "italic": False, "text": lore[1]},
                                    {"color": "gray", "italic": False, "text": lore[2]},
                                    {"color": "gray", "italic": False, "text": lore[3]},
                                ],
                                "mode": "append"
                            }
                        ]
                    }
                ]
            }
        ]
    }

    with open(CARD_FOLDER / f"{ident}.json", "w", encoding="utf-8") as f:
        json.dump(card_json, f, indent=2, ensure_ascii=False)

    print(f"Generated card {ident}.json")

# ----------------------------------------------------------------
# WRITE give_all_cards.mcfunction
# ----------------------------------------------------------------
FUNC_FILE = FUNCTION_FOLDER / "give_all_cards.mcfunction"

with open(FUNC_FILE, "w", encoding="utf-8") as f:
    for entry in all_cards:
        f.write(f"loot give @s loot {entry}\n")

print(f"Created {FUNC_FILE}")

# ----------------------------------------------------------------
# RANDOM LOOT TABLES
# ----------------------------------------------------------------
def write_random_table(filename, items):
    data = {
        "pools": [
            {
                "rolls": 1,
                "entries": [
                    {"type": "loot_table", "weight": 1, "value": item}
                    for item in items
                ]
            }
        ]
    }
    with open(RAND_FOLDER / (filename + ".json"), "w", encoding="utf-8") as f:
        json.dump(data, f, indent=2)
    print(f"Generated random table {filename}.json")


write_random_table("give_random_card_any", all_cards)
write_random_table("give_random_card_common", common_cards)
write_random_table("give_random_card_rare", rare_cards)
write_random_table("give_random_card_legendary", legendary_cards)

print("Done.")
